---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
function GetPercentageAt( Value, ValueAt, bRound )
	local res = ( ValueAt ~= 0 and Value / ValueAt or 0 ) * 100
	return bRound and math.floor( res ) or res
end
--------------------------------------------------------------------------------
function GetSpellInfoNameFromParams( params )
	if params then
		local ID = params.spellId or params.buffId
		local info = params.spellId and avatar.GetSpellInfo( ID )
			or params.buffId and avatar.GetBuffInfoById( ID )
		
		if info and not info.debugName then
			info.debugName = userMods.FromWString( info.name )
		end
		
		return info, ID
	end
end
--------------------------------------------------------------------------------
function IsThisStringValue( value1, value2 )
	if common.IsWString( value1 ) and common.IsWString( value2 ) then
		return common.CompareWString( value1, value2 ) == 0
	else
		return type( value1 ) == "string" and value1 == value2
	end
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
Global( "TDamageDetails", {} )
---------------------------------------------------------------------------------------------------------------------------
function TDamageDetails:CreateNewObject( Name )
	return setmetatable( {
		Name = Name,
		Type = "",
		Count = 0,
		Percentage = 0,
		DamageAmount = 0,
		Min = -1,
		Mid = -1,
		Max = -1,
	}, { __index = self } )
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
Global( "TSpellDamageData", {} )
---------------------------------------------------------------------------------------------------------------------------
function TSpellDamageData:CreateNewObject()
	return setmetatable( {
		Name = "",
		ID = 0,
		DebugName = "",
		DamageType = "",
		DamageAmount = 0,
		DPS = 0,
		DamagePercentage = 0,
		DamageDetailsList = {
			TDamageDetails:CreateNewObject( "HIT" ),
			TDamageDetails:CreateNewObject( "CRIT" ),
			TDamageDetails:CreateNewObject( "DODGE" ),
			TDamageDetails:CreateNewObject( "GLANCING" ),
			TDamageDetails:CreateNewObject( "MISS" )
		}
	}, { __index = self } )
end
--------------------------------------------------------------------------------
function TSpellDamageData:ReceiveValuesFromParams( params )
	local DamageType
	
	if params.isCritical == true then
		DamageType = "CRIT"
	elseif params.isGlancing == true then
		DamageType = "GLANCING"
	elseif params.isMiss == true then
		DamageType = "MISS"
	elseif params.isDodge == true then
		DamageType = "DODGE"
	else
		DamageType = "HIT"
	end

	local DamageDetails = self:GetDamageDetailByName( DamageType )
	
	DamageDetails.Count = DamageDetails.Count + 1
	DamageDetails.DamageAmount = DamageDetails.DamageAmount + params.amount
	
	if ( params.amount < DamageDetails.Min )  or ( DamageDetails.Min == -1 ) then
		DamageDetails.Min = params.amount
	end
	if ( params.amount > DamageDetails.Max )  or ( DamageDetails.Max == -1 ) then
		DamageDetails.Max = params.amount
	end
	
	DamageDetails.Mid = math.floor( (DamageDetails.Min + DamageDetails.Max ) / 2 )
	
end
--------------------------------------------------------------------------------
function TSpellDamageData:GetDamageAmount()
	local res = 0
	for i, DamageDetail in self.DamageDetailsList do
		res = res + DamageDetail.DamageAmount
	end
	return res
end
--------------------------------------------------------------------------------
function TSpellDamageData:GetUseCount()
	local res = 0
	for i, DamageDetail in self.DamageDetailsList do
		res = res + DamageDetail.Count
	end
	return res
end
--------------------------------------------------------------------------------
function TSpellDamageData:GetDamageDetailByName( Name )
	for i, DamageDetail in self.DamageDetailsList do
		if IsThisStringValue( DamageDetail.Name, Name ) then
			return DamageDetail
		end
	end
	return nil
end
--------------------------------------------------------------------------------
local function CompareDamageDetails( v1, v2 )
	return v1.Count > v2.Count
end
--------------------------------------------------------------------------------
function TSpellDamageData:SortSpellDetailsByCount()
	table.sort( self.DamageDetailsList, CompareDamageDetails )
end
--------------------------------------------------------------------------------
function TSpellDamageData:CalculateSpellDetailsPercentage()
	local Count = self:GetUseCount()
	for i, DamageDetail in  self.DamageDetailsList do
		DamageDetail.Percentage = GetPercentageAt( DamageDetail.Count, Count, true )
	end
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
Global( "TCombatant", {} )
---------------------------------------------------------------------------------------------------------------------------
function TCombatant:CreateNewObject( ID )
	return setmetatable( {
		Class = unit.GetClass( ID ).className,
		Name = object.GetName( ID ),
		ID = ID,
		DamageAmount = 0,
		DPS = 0,
		DamagePercentage = 0,
		bCanUpdate = true,
		SpellsList = {},
	}, { __index = self } )
end
--------------------------------------------------------------------------------
function TCombatant:CreateNewObjectByName( Name )
	return setmetatable( {
		Class = "PRIEST",
		Name = Name,
		ID = nil,
		DamageAmount = 0,
		DPS = 0,
		DamagePercentage = 0,
		bCanUpdate = false,
		SpellsList = {},
	}, { __index = self } )
end
--------------------------------------------------------------------------------
function TCombatant:GetSpellByNameDebug( DebugName )
	for i, Spell in self.SpellsList do 
		if Spell.DebugName == DebugName then
			return Spell
		end
	end
	return nil
end
--------------------------------------------------------------------------------
function TCombatant:GetSpellByName( Name )
	for i, Spell in self.SpellsList do 
		if IsThisStringValue( Spell.Name, Name ) then
			return Spell
		end
	end
	return nil
end
--------------------------------------------------------------------------------
function TCombatant:GetSpellByIndex( Index )
	return self.SpellsList[ Index + 1 ]
end
--------------------------------------------------------------------------------
function TCombatant:AddNewSpell( params, Name )
	local SpellInfo, ID = GetSpellInfoNameFromParams( params )
	
	if SpellInfo and ID then 
		local Spell = TSpellDamageData:CreateNewObject()
		Spell.Name = Name or params.ability
		Spell.DebugName = SpellInfo.debugName
		Spell.DamageType = params.sysSubElement
		Spell.ID = ID
		
		table.insert( self.SpellsList, Spell )
		return Spell
	end
end
--------------------------------------------------------------------------------
function TCombatant:ClearSpellData()
	self.SpellsList = {}
end
--------------------------------------------------------------------------------
function TCombatant:CalculateSpellDamage( FightTime )
	for i, Spell in self.SpellsList do 
		Spell.DamageAmount = Spell:GetDamageAmount()
		Spell.DPS = Spell.DamageAmount / FightTime
		Spell.DamagePercentage = GetPercentageAt( Spell.DamageAmount, self.DamageAmount, true )
		Spell:CalculateSpellDetailsPercentage()
		Spell:SortSpellDetailsByCount()
	end
end
--------------------------------------------------------------------------------
local function CompareSpells( v1, v2 )
	return v1.DamageAmount > v2.DamageAmount
end
--------------------------------------------------------------------------------
function TCombatant:SortSpellByDamageAmount()
	table.sort( self.SpellsList, CompareSpells )
end
--------------------------------------------------------------------------------
function TCombatant:UpdateCombatantDataByID( ID )
	self.ID = ID
	if ID then
		self.Class = unit.GetClass( ID ).className
		self.Name = object.GetName( ID )
	else
		self.Class = "PRIEST"
	end
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
Global( "TFight", {} )
Global( "bForceSetCombatMode", false )
---------------------------------------------------------------------------------------------------------------------------
function TFight:CreateNewObject( Name )
	return setmetatable( {
		Name = Name,
		FightTime = 1,
		DamageAmount = 0,
		CombatantsList = {},
	}, { __index = self } )
end
---------------------------------------------------------------------------------------------------------------------------
function TFight:AddNewCombatant( ID )
	if not self:GetCombatantByName( object.GetName( ID ) ) then
		table.insert( self.CombatantsList, TCombatant:CreateNewObject( ID ) )
	end
end
--------------------------------------------------------------------------------
function TFight:AddNewCombatantByName( Name )
	if not self:GetCombatantByName( Name ) then
		table.insert( self.CombatantsList, TCombatant:CreateNewObjectByName( Name ) )
	end
end
--------------------------------------------------------------------------------
function TFight:RemoveCombatantByName( Name )
	local Combatant, index = self:GetCombatantByName( Name )
	
	if index then
		table.remove( self.CombatantsList, index )
	end
end
--------------------------------------------------------------------------------
function TFight:GetCombatantByName( Name )
	for i, Combatant in self.CombatantsList do 
		if IsThisStringValue( Combatant.Name, Name ) then
			return Combatant, i
		end
	end
	return nil
end
--------------------------------------------------------------------------------
function TFight:GetCombatantByID( ID )
	for i, Combatant in self.CombatantsList do 
		if Combatant.bCanUpdate then 
			if Combatant.ID == ID then
				return Combatant
			end
		end
	end
	return nil
end
--------------------------------------------------------------------------------
function TFight:GetCombatantByIndex( Index )
	return self.CombatantsList[ Index ]
end
--------------------------------------------------------------------------------
function TFight:GetCombatantNameByIndex( Index )
	local Combatant = self.CombatantsList[ Index + 1 ]
	return Combatant and Combatant.Name or ""
end
--------------------------------------------------------------------------------
function TFight:GetCombatantCount()
	return table.getn( self.CombatantsList )
end
---------------------------------------------------------------------------------------------------------------------------
function TFight:GetCombatantDamageData( Combatant, params )
	if Combatant.bCanUpdate then
		if (params.source) and ( params.source == Combatant.ID ) and ( params.target ~= Combatant.ID ) then

			if (params.amount) and ( not params.isFall )  then 
				
				Combatant.DamageAmount = Combatant.DamageAmount + params.amount
				Combatant.DPS = Combatant.DamageAmount / self.FightTime
				
				local SpellInfo, ID = GetSpellInfoNameFromParams( params )
				
				if SpellInfo then
					local Spell = Combatant:GetSpellByNameDebug( SpellInfo.debugName )

					if( Spell == nil ) then 
						Spell = Combatant:AddNewSpell( params )
					end
					
					if Spell then Spell:ReceiveValuesFromParams( params ) end
				end
				
			end

		elseif (params.source) and ( unit.IsPet( params.source ) ) and ( unit.GetPetOwner( params.source ) ==  Combatant.ID ) then
			
			if ( params.amount ) then 
				Combatant.DamageAmount = Combatant.DamageAmount + params.amount
				Combatant.DPS = Combatant.DamageAmount / self.FightTime
				
				local PetName = object.GetName( params.source )
				
				local Spell = Combatant:GetSpellByName( PetName )
				
				if( Spell == nil ) then 
					Spell = Combatant:AddNewSpell( params, PetName )
				end
				if Spell then 
					Spell:ReceiveValuesFromParams( params )
					Spell.DamageType = "ENUM_SubElement_PHYSICAL"
				end
				
			end	
			
		end
	end

end
---------------------------------------------------------------------------------------------------------------------------
function TFight:CalculateDamage()
	self.DamageAmount = 0
	for i, Combatant in self.CombatantsList do 
		self.DamageAmount = self.DamageAmount + Combatant.DamageAmount 
	end
end
--------------------------------------------------------------------------------
function TFight:UpdateCombatantsData( params )
	if params then 
		for i, Combatant in self.CombatantsList do 
			if Combatant.bCanUpdate then
				self:GetCombatantDamageData( Combatant, params )
			end
		end
	end
	
	self:CalculateDamage()
	
	for i, Combatant in self.CombatantsList do 
		Combatant.DamagePercentage = GetPercentageAt( Combatant.DamageAmount, self.DamageAmount, true )
		Combatant:CalculateSpellDamage( self.FightTime )
		Combatant:SortSpellByDamageAmount()
	end
	
	self:SortCombatantsByDamageAmount()
	
end
--------------------------------------------------------------------------------
function TFight:ClearCombatantsData()
	for i, Combatant in self.CombatantsList do 
		Combatant.DamageAmount = 0
		Combatant.DPS = 0
		Combatant.DamagePercentage = 0
		Combatant:ClearSpellData()
	end
end
--------------------------------------------------------------------------------
function TFight:IsCombatantsInCombat()
	if bForceSetCombatMode then 
		return true
	end
	
	for i, Combatant in self.CombatantsList do 
		if Combatant.bCanUpdate and unit.IsInCombat( Combatant.ID ) then
			return true
		end
	end
end
--------------------------------------------------------------------------------
function TFight:IsCombatantsInCombatWithoutForceSet()
	for i, Combatant in self.CombatantsList do 
		if Combatant.bCanUpdate and unit.IsInCombat( Combatant.ID ) then
			return true
		end
	end
end
--------------------------------------------------------------------------------
local function CompareCombatantsByDamageAmount( v1, v2 )
	return v1.DamageAmount > v2.DamageAmount
end
--------------------------------------------------------------------------------
function TFight:SortCombatantsByDamageAmount()
	table.sort( self.CombatantsList, CompareCombatantsByDamageAmount )
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
Global( "TDPSMeter", {} )
---------------------------------------------------------------------------------------------------------------------------
function TDPSMeter:CreateNewObject()
	return setmetatable( {
		FightsList = {},
		bCollectData = false,
		ResetFightName = "",
	}, { __index = self } )
end
--------------------------------------------------------------------------------
function TDPSMeter:AddNewFight( Name )
	table.insert( self.FightsList, TFight:CreateNewObject( Name ) )
end
--------------------------------------------------------------------------------
function TDPSMeter:GetFightByName( Name )
	for i, Fight in self.FightsList do 
		if IsThisStringValue( Fight.Name, Name ) then
			return Fight
		end
	end
	return nil
end
--------------------------------------------------------------------------------
function TDPSMeter:AddNewCombatant( ID )
	for i, Fight in self.FightsList do 
		Fight:AddNewCombatant( ID )
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:SetCombatantUpdateStatusByName( Name, bCanUpdate )
	for i, Fight in self.FightsList do 
		local Combatant = Fight:GetCombatantByName( Name )
		
		if Combatant then
			Combatant.bCanUpdate = bCanUpdate;
		end
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:AddNewCombatantByName( Name )
	for i, Fight in self.FightsList do 
		Fight:AddNewCombatantByName( Name )
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:RemoveAllCombatants()
	for i, Fight in self.FightsList do 
		Fight.CombatantsList = {}
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:GetAllCombatants()
	return self.FightsList[ 1 ].CombatantsList
end
--------------------------------------------------------------------------------
function TDPSMeter:RemoveCombatantByName( Name )
	for i, Fight in self.FightsList do 
		Fight:RemoveCombatantByName( Name )
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:IsCombatantsInCombatWithoutForceSet()
	for i, Fight in self.FightsList do 
		if Fight:IsCombatantsInCombatWithoutForceSet() then 
			return true
		end
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:IsParamsHaveCombatants( params )
	if not params then
		return false
	end

	for i, Fight in self.FightsList do 
		for j, Combatant in Fight.CombatantsList do 
			--if Combatant.bCanUpdate and ( Combatant.ID == params.source or ( ( unit.IsPet(params.source) ) and ( unit.GetPetOwner(params.source) == Combatant.ID) ) ) then
			if Combatant.bCanUpdate and Combatant.ID == params.source then
				return true
			end
		end
	end

	return false
end
--------------------------------------------------------------------------------
function TDPSMeter:UpdateFightsData( params )
	if not self.bCollectData
	and self:IsParamsHaveCombatants( params ) then
		bForceSetCombatMode = true
		self.bCollectData = true
		self:ResetFightByName( self.ResetFightName )
		
	elseif self:IsCombatantsInCombatWithoutForceSet() then
		bForceSetCombatMode = false
	end
	
	for i, Fight in self.FightsList do 
		 Fight:UpdateCombatantsData( params )
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:UpdateFightsTime()
	for i, Fight in self.FightsList do 
		 Fight.FightTime = Fight.FightTime + 1
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:ResetAllFights()
	for i, Fight in self.FightsList do 
		 Fight.FightTime = 1
		 Fight.DamageAmount = 0
		 Fight:ClearCombatantsData()
	end
end
--------------------------------------------------------------------------------
function TDPSMeter:ResetFightByName( Name )
	local Fight = self:GetFightByName( Name )
	Fight.FightTime = 1
	Fight.DamageAmount = 0
	Fight:ClearCombatantsData()
end
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------------------------------
